import fitz
import cv2
import numpy as np


class ImageLoader:

    """Extract an image or list of images from a file.

       loader = ImageLoader("some_file.pdf")
       loader.images will be the list of images from the file
    """

    def __init__(self, fname):
        if fname.endswith(".pdf"):
            self._load_pdf_images(fname)
        else:
            self._load_single_image(fname)

    def _load_pdf_images(self, fname):
        pdf = fitz.open(fname)
        pixmaps = [pdf.getPagePixmap(pagenum, alpha=False)
                   for pagenum in range(len(pdf))]
        self.images = [self._pixmap_to_image(pm) for pm in pixmaps]

    def _pixmap_to_image(self, pix):
        im = np.frombuffer(pix.samples, np.uint8).reshape(pix.h, pix.w, pix.n)
        im = np.ascontiguousarray(im[..., [2, 1, 0]])
        return im

    def _load_single_image(self, filename):
        self.images = [cv2.imread(filename)]
